/* Michael Batchelder and Kacper Wysocki
 * McGill COMP-767 Non-photorealistic rendering
 *
 * Based on NPR Quake by Alex Mohr and others,
 * and code by Adrian Ilie.
 * www.cs.unc.edu/~adyilie/comp238/Final/Final.htm
 * http://www.cs.wisc.edu/graphics/Gallery/NPRQuake/)
 * References: Ramesh Raskar, Gooch et al.
 */

#include <math.h>
#include <float.h>
#include "quakedef.h"
#include "dynamic_r.h"
#include "time.h"
#include "matrixFunc.h"

#ifndef _WIN32
#include "hash3d.h"
#endif
typedef GLfloat Vec3[3];

// this is a sine lookup table for doing vertex permutations for water/teleporters..
// - Alex
static float	turbsin[] =
{
	#include "dr_warp_sin.h"
};

#define NUMVERTEXNORMALS 162
static float norms [NUMVERTEXNORMALS][3] =
{
	#include "anorms.h"
};

#ifndef _WIN32
#define isNaNm(v) (isnan(v[0]) || isnan(v[1]) || isnan(v[2]))
#endif

//sil modes
#define SIL_NONE 0
#define SIL_FLAT 1
#define SIL_SHADED 2

//wall sketch stuff
#define RAND_TABLE_SIZE (1024)
#define SHADOW_HEIGHT (20)
// this contains random floats in the range [0..1) -- it's used to generate random
// numbers without calling rand all the time.
static float RandTable[RAND_TABLE_SIZE];
// this is the 'current random number'
static unsigned int CurRand;

static cvar_t tech_shadows = { "tech_shadows", "1" };
static cvar_t tech_shadow_offset = { "tech_shadow_offset", "4" };
static cvar_t tech_shadow_color = { "tech_shadow_color", "0x00008F" };
static cvar_t tech_shadow_scale = { "tech_shadow_scale", "1.3" };
static cvar_t tech_shadow_umbra = { "tech_shadow_umbra", "0.85" };
static cvar_t tech_shadow_alpha = { "tech_shadow_alpha", "0.5" };
static cvar_t tech_shade_warm = { "tech_shade_warm", "0x808000" };
static cvar_t tech_shade_cool = { "tech_shade_cool", "0x0000FF" };
static cvar_t tech_shade_model = { "tech_shade_model", "1"};
static cvar_t tech_shade_model_mode = { "tech_shade_model_mode", "1"};
static cvar_t tech_color_models = { "tech_color_models","0"};
// antialias crease/sil lines
static cvar_t tech_antialias = { "tech_antialias", "1" };
//object silhouettes
static cvar_t tech_sil_width = { "tech_sil_width", "4" };
static cvar_t tech_sil_mode = { "tech_sil_mode", "1" };
static cvar_t tech_sil_color = { "tech_sil_color", "0x000000" };
static cvar_t tech_sil_depth = { "tech_sil_depth", "0" };

// creases
static cvar_t tech_crease_mode = { "tech_crease_mode", "1" };
static cvar_t tech_crease_depth = { "tech_crease_depth", "0" };
static cvar_t tech_crease_color = { "tech_crease_color", "0xFFFFFF" };
static cvar_t tech_crease_width = { "tech_crease_width", "2" };
static cvar_t tech_color_thresh = { "tech_color_thresh", "0.1" };
//walls
static cvar_t tech_line_width = { "tech_line_width", "3" };
static cvar_t tech_wall_color = { "tech_wall_color", "0x888800" };
static cvar_t tech_wall_line_color = { "tech_wall_line_color", "0x332415" };
static cvar_t tech_water_color = { "tech_water_color", "0x595113" };

//other
//global vars
static float gun_pos[4];

static void (*dr_ConPrintf)( char * fmt, ... ) = NULL;
static void (*dr_VectorMA)( vec3_t, float, vec3_t, vec3_t ) = NULL;
static void (*dr_GL_Bind)( int ) = NULL;
static void (*dr_GL_DisableMultitexture)() = NULL;
static void (*dr_GL_EnableMultitexture)() = NULL;
static void (*dr_AngleVectors)( vec3_t, vec3_t, vec3_t, vec3_t ) = NULL;
static texture_t * (*dr_R_TextureAnimation)( texture_t * ) = NULL;
static void (*dr_VectorScale)( vec3_t, vec_t, vec3_t ) = NULL;
static void (*dr_RegisterVar)( cvar_t * var );
static void (*dr_UnRegisterVar)( cvar_t * var );

//static float idvec[] = {0,0,0,1};

#ifndef _WIN32
/* broken math.h, lacking prototypes */
float fabsf(float x);
float powf(float x, float y);
#endif

void printVert(float *v){
	dr_ConPrintf("(%f,%f,%f)\n",v[0],v[1],v[2]);
}

void printVertU(u_char *v){
	dr_ConPrintf("(%d,%d,%d)",v[0],v[1],v[2]);
}

void printDist(float *v){
	dr_ConPrintf("   %f\n",sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2]));
}


#ifndef _WIN32
int checkNorm(float *norm){
	if(isNaNm(norm))
		return 1;
	return 0;
}
#endif

#ifndef _WIN32
int assertEdge(edge_t *e) {
	if (e==NULL) {
		dr_ConPrintf("Empty Edge, ASSHOLE");
		return 1;
	}
	if (e->v1 == NULL || e->v2 == NULL) {
		dr_ConPrintf("Vert is NULL");
		return 1;
	}
	if (e->n1 == NULL) {
		dr_ConPrintf("Normal is NULL");
		return 1;
	}
	if (isNaNm(e->v1) || isNaNm(e->v2)) {
		dr_ConPrintf("Vert is not a number");
		return 1;
	}
	if (isNaNm(e->n1)) {
		dr_ConPrintf("Normal1 is not a number");
		return 1;
	}
	if (e->n2 == NULL) {
		dr_ConPrintf("\t\t\tN2 is NULL\n");
		if (!(e->rn & 1))
			dr_ConPrintf("\t\t\tEDGE IS SINGULAR\n");
	} else if (isNaNm(e->n2)) {
		dr_ConPrintf("Normal2 is not a number");
		return 1;
	}

	return 0;
}
#endif

//copied over from common.c to parse hex strings
float Q_atof (char *str) {
	double			val;
	int			 sign;
	int			 c;
	int			 decimal, total;

	if (*str == '-')
	{
		sign = -1;
		str++;
	}
	else
		sign = 1;

	val = 0;

//
// check for hex
//
	if (str[0] == '0' && (str[1] == 'x' || str[1] == 'X') )
	{
		str += 2;
		while (1)
		{
			c = *str++;
			if (c >= '0' && c <= '9')
				val = (val*16) + c - '0';
			else if (c >= 'a' && c <= 'f')
				val = (val*16) + c - 'a' + 10;
			else if (c >= 'A' && c <= 'F')
				val = (val*16) + c - 'A' + 10;
			else
				return val*sign;
		}
	}

//
// check for character
//
	if (str[0] == '\'')
	{
		return sign * str[1];
	}

//
// assume decimal
//
	decimal = -1;
	total = 0;
	while (1)
	{
		c = *str++;
		if (c == '.')
		{
			decimal = total;
			continue;
		}
		if (c <'0' || c > '9')
			break;
		val = val*10 + c - '0';
		total++;
	}

	if (decimal == -1)
		return val*sign;
	while (total > decimal)
	{
		val /= 10;
		total--;
	}

	return val*sign;
}


// set GL color from a hex float
void setColor(float color) {
	float r,g,b;
	int c;
	c = (int)color;
	b = (c % 256)/256.0;
	c /= 256;
	g = (c % 256)/256.0;
	r = (c / 256)/256.0;
	glColor3f(r,g,b);
}


// set GL color from a hex float multiplied by a lighting coefficient
void setColor2(float color, float l) {
	float r,g,b;
	int c;
	c = (int)color;
	b = (c % 256)/256.0*l;
	c /= 256;
	g = (c % 256)/256.0*l;
	r = (c / 256)/256.0*l;
	glColor3f(r,g,b);
}

void getColor(float color, float *l) {
	float r,g,b;
	int c;
	c = (int)color;
	b = (c % 256)/256.0;
	c /= 256;
	g = (c % 256)/256.0;
	r = (c / 256)/256.0;
	l[0] = r; l[1] = g; l[2] = b;
}


//set a color vector from a hex float
void texColor(float color, float *c) {
	int cl;
	cl = (int)color;
	c[2] = (cl % 256)/256.0;
	cl /= 256;
	c[1] = (cl % 256)/256.0;
	c[0] = (cl / 256)/256.0;
}

/*
================
triNorm -- takes a vertex pointer to a triangle's first vertex and a float array.
The normal to the triangle at the three sequential vertices is returned in norm.
================
*/
//copied over form sketch NPR Quake
void triNorm (u_char index, float *vert1, float *vert2, float *vert3, float *norm) {
	float	length;
	vec3_t	v1, v2;

	VectorSubtract(vert2, vert1, v1);
	VectorSubtract(vert3, vert1, v2);

	norm[0] = v1[1]*v2[2] - v1[2]*v2[1];
	norm[1] = v1[2]*v2[0] - v1[0]*v2[2];
	norm[2] = v1[0]*v2[1] - v1[1]*v2[0];

	// make sure normal agrees with vertex light normal!
	if( (norm[0]*norms[index][0] + norm[1]*norms[index][1] + norm[2]*norms[index][2]) < 0) {
		norm[0] = -norm[0];
		norm[1] = -norm[1];
		norm[2] = -norm[2];
	}

	length = sqrt(norm[0]*norm[0] + norm[1]*norm[1] + norm[2]*norm[2]);
	norm[0] /= length;
	norm[1] /= length;
	norm[2] /= length;
}

// IMMEDIATELY after that, it should call this Init function.
EXPORT void dr_Init() {

	int i;

	// make sure GL's state is kept
	glPushAttrib(GL_ALL_ATTRIB_BITS);
	dr_ConPrintf( "Technical NPR renderer starting up...\n" );

	dr_ConPrintf( "registering my vars\n" );

	dr_RegisterVar(&tech_shadow_alpha);
	dr_RegisterVar(&tech_shadow_color);
	dr_RegisterVar(&tech_shadow_scale);
	dr_RegisterVar(&tech_shadow_umbra);
	dr_RegisterVar(&tech_shadows);
	dr_RegisterVar(&tech_wall_color);
	dr_RegisterVar(&tech_wall_line_color);
	dr_RegisterVar(&tech_water_color);
	dr_RegisterVar(&tech_shadow_offset);
	dr_RegisterVar(&tech_shade_cool);
	dr_RegisterVar(&tech_shade_warm);
	dr_RegisterVar(&tech_shade_model);
	dr_RegisterVar(&tech_shade_model_mode);
	dr_RegisterVar(&tech_color_models);

	dr_RegisterVar(&tech_antialias);
	dr_RegisterVar(&tech_sil_width);
	dr_RegisterVar(&tech_sil_mode);
	dr_RegisterVar(&tech_sil_color);
	dr_RegisterVar(&tech_sil_depth);
	dr_RegisterVar(&tech_crease_mode);
	dr_RegisterVar(&tech_crease_color);
	dr_RegisterVar(&tech_crease_width);
	dr_RegisterVar(&tech_crease_depth);
	dr_RegisterVar(&tech_color_thresh);

	dr_RegisterVar( &tech_line_width );

	// seed the generator with the time.
	srand( 1000 );
	for( i = 0; i < RAND_TABLE_SIZE; i++ ) {
		RandTable[i] = (float)rand() / (float)RAND_MAX;
	}

	// current random number is the first one;
	CurRand = 0;

	// initialize our neighbouring information model hash
#ifndef _WIN32
	init_modelhash();
#endif
}


// BEFORE removing the renderer, it should call this Shutdown function.
EXPORT void dr_Shutdown() {

	dr_ConPrintf( "Technical NPR renderer shutting down...\n" );

	dr_ConPrintf( "unregistering my vars\n" );

	dr_UnRegisterVar(&tech_shadow_alpha);
	dr_UnRegisterVar(&tech_shadow_color);
	dr_UnRegisterVar(&tech_shadow_scale);
	dr_UnRegisterVar(&tech_shadow_umbra);
	dr_UnRegisterVar(&tech_shadows);
	dr_UnRegisterVar(&tech_wall_color);
	dr_UnRegisterVar(&tech_wall_line_color);
	dr_UnRegisterVar(&tech_water_color);
	dr_UnRegisterVar(&tech_shadow_offset);
	dr_UnRegisterVar(&tech_shade_warm);
	dr_UnRegisterVar(&tech_shade_cool);
	dr_UnRegisterVar(&tech_shade_model);
	dr_UnRegisterVar(&tech_shade_model_mode);
	dr_UnRegisterVar(&tech_color_models);
	dr_UnRegisterVar(&tech_antialias);
	dr_UnRegisterVar(&tech_sil_width);
	dr_UnRegisterVar(&tech_sil_mode);
	dr_UnRegisterVar(&tech_sil_color);
	dr_UnRegisterVar(&tech_sil_depth);
	dr_UnRegisterVar(&tech_crease_mode);
	dr_UnRegisterVar(&tech_crease_color);
	dr_UnRegisterVar(&tech_crease_width);
	dr_UnRegisterVar(&tech_crease_depth);
	dr_UnRegisterVar(&tech_color_thresh);
	dr_UnRegisterVar( &tech_line_width );

	// make sure gl's state is consistent - pop off from dr_Init()
	glPopAttrib();
}


// random numbers for the sketch lines on the walls and water (copied over from sketch NPR Quake)
float randFloat (float min, float max)
{
	// I know this seems backwards, but it works and avoids an extra variable.
	if( CurRand >= RAND_TABLE_SIZE ) {
		CurRand = 0;
	}
	return( RandTable[CurRand++] * (max - min) + min );
}


// The only non-export function.

/*
================
R_GetSpriteFrame
================
*/
mspriteframe_t *R_GetSpriteFrame (entity_t *currententity, double cltime)
{
	msprite_t		*psprite;
	mspritegroup_t	*pspritegroup;
	mspriteframe_t	*pspriteframe;
	int				i, numframes, frame;
	float			*pintervals, fullinterval, targettime, time;

	psprite = currententity->model->cache.data;
	frame = currententity->frame;

	if ((frame >= psprite->numframes) || (frame < 0))
	{
		dr_ConPrintf ("R_DrawSprite: no such frame %d\n", frame);
		frame = 0;
	}

	if (psprite->frames[frame].type == SPR_SINGLE)
	{
		pspriteframe = psprite->frames[frame].frameptr;
	}
	else
	{
		pspritegroup = (mspritegroup_t *)psprite->frames[frame].frameptr;
		pintervals = pspritegroup->intervals;
		numframes = pspritegroup->numframes;
		fullinterval = pintervals[numframes-1];

		time = cltime + currententity->syncbase;

	// when loading in Mod_LoadSpriteGroup, we guaranteed all interval values
	// are >= 1, so we don't have to worry about division by 0
		targettime = time - ((int)(time / fullinterval)) * fullinterval;

		for (i=0 ; i<(numframes-1) ; i++)
		{
			if (pintervals[i] > targettime)
				break;
		}

		pspriteframe = pspritegroup->frames[i];
	}

	return pspriteframe;
}


/********************************************************************************/

EXPORT void dr_Set_Cvar_RegisterVariable( void (*f)( cvar_t * ) ) {
	dr_RegisterVar = f;
}
EXPORT void dr_Set_Cvar_UnRegisterVariable( void (*f)( cvar_t * ) ) {
	dr_UnRegisterVar = f;
}
EXPORT void dr_Set_ConPrintf( void (*f)( char * fmt, ... ) ) {
	dr_ConPrintf = f;
}
EXPORT void dr_Set_VectorMA( void (*f)( vec3_t, float, vec3_t, vec3_t ) ) {
	dr_VectorMA = f;
}
EXPORT void dr_Set_GL_Bind( void (*f)( int ) ) {
	dr_GL_Bind = f;
}
EXPORT void dr_Set_GL_DisableMultitexture( void (*f)() ) {
	dr_GL_DisableMultitexture = f;
}
EXPORT void dr_Set_GL_EnableMultitexture( void (*f)() ) {
	dr_GL_EnableMultitexture = f;
}
EXPORT void dr_Set_AngleVectors( void (*f)( vec3_t, vec3_t, vec3_t, vec3_t ) ) {
	dr_AngleVectors = f;
}
EXPORT void dr_Set_R_TextureAnimation( texture_t * (*f)( texture_t * ) ) {
	dr_R_TextureAnimation = f;
}
EXPORT void dr_Set_VectorScale( void (*f)( vec3_t, vec_t, vec3_t ) ) {
	dr_VectorScale = f;
}

/********************************************************************************/

static GLenum oldtarget = TEXTURE0_SGIS;

EXPORT void GL_SelectTexture (GLenum target, qboolean gl_mtexable,
					   lpSelTexFUNC qglSelectTextureSGIS, int currenttexture,
					   int * cnttextures) {
	if (!gl_mtexable)
		return;
	qglSelectTextureSGIS(target);
	if (target == oldtarget)
		return;
	cnttextures[oldtarget-TEXTURE0_SGIS] = currenttexture;
	currenttexture = cnttextures[target-TEXTURE0_SGIS];
	oldtarget = target;
}

/*
=================
R_DrawSpriteModel

=================
*/
EXPORT void R_DrawSpriteModel (entity_t *e, entity_t *currententity,
						double cltime, vec3_t vup, vec3_t vright) {
	vec3_t	point;
	mspriteframe_t	*frame;
	float		*up, *right;
	vec3_t		v_forward, v_right, v_up;
	msprite_t		*psprite;

	// don't even bother culling, because it's just a single
	// polygon without a surface cache
	frame = R_GetSpriteFrame( e, cltime );
	psprite = currententity->model->cache.data;

	if (psprite->type == SPR_ORIENTED)
	{	// bullet marks on walls
		dr_AngleVectors (currententity->angles, v_forward, v_right, v_up);
		up = v_up;
		right = v_right;
	}
	else
	{	// normal sprite
		up = vup;
		right = vright;
	}

	glColor3f (1,1,1);

	dr_GL_DisableMultitexture();

	dr_GL_Bind(frame->gl_texturenum);

	glEnable (GL_ALPHA_TEST);
	glBegin (GL_QUADS);

	glTexCoord2f (0, 1);
	dr_VectorMA (e->origin, frame->down, up, point);
	dr_VectorMA (point, frame->left, right, point);
	glVertex3fv (point);

	glTexCoord2f (0, 0);
	dr_VectorMA (e->origin, frame->up, up, point);
	dr_VectorMA (point, frame->left, right, point);
	glVertex3fv (point);

	glTexCoord2f (1, 0);
	dr_VectorMA (e->origin, frame->up, up, point);
	dr_VectorMA (point, frame->right, right, point);
	glVertex3fv (point);

	glTexCoord2f (1, 1);
	dr_VectorMA (e->origin, frame->down, up, point);
	dr_VectorMA (point, frame->right, right, point);
	glVertex3fv (point);

	glEnd ();

	glDisable (GL_ALPHA_TEST);
}

#ifndef _WIN32
void printEdge(edge_t *e) {
	if (e==NULL)
		dr_ConPrintf("You asked me to print a NULL edge, ASSHOLE!");
	else {
		if (e->v1 == NULL || e->v2 == NULL)
		{
			dr_ConPrintf("You have null Vertices, Idiot");
			return;
		}
		//dr_ConPrintf("%f, %f, %f   %f, %f, %f\n",
		//	e->v1[0],e->v1[1],e->v1[2],
		//	e->v2[0],e->v2[1],e->v2[2]);
		if (e->n2==NULL)
			dr_ConPrintf("\t%f, %f, %f   NULL  NULL  NULL\n",
				e->n1[0],e->n1[1],e->n1[2]);
		else
			dr_ConPrintf("\t%f, %f, %f   %f, %f, %f\n",
				e->n1[0],e->n1[1],e->n1[2],
				e->n2[0],e->n2[1],e->n2[2]);
	}
}
#endif

/*
void print_edge_table(aliashdr_t *paliashdr, int posenum){
	trivertx_t	*verts;
	int		*order;
	int		count;
	int		total = 0;
	int 		size = 0;
	int		last;
	trivertx_t	*list = NULL;
	int		*countl = NULL;

	dr_ConPrintf("Printing Edge Table... \n");
	verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);
	while (1)
	{
		count = abs(*order++);
		if (count==0) break;

		while (count-->0) {
			int i, add = 1;
			for (i = 0; i < size; i++) {
				if (memcmp(&(list[i]),verts,sizeof(byte)*3)==0)
				{
					if (memcmp(&(list[i]),verts,sizeof(trivertx_t))!=0) {
						dr_ConPrintf("YUPYUPYUP\n\n\n\n\n\n\n\n");
					} else {
						add = 0;
						countl[i]++;
						break;
					}
				}
			}
			if (add==1) {
				list = realloc(list,sizeof(trivertx_t)*(size+1));
				countl = realloc(countl,sizeof(int)*(size+1));
				countl[size] = 1;
				list[size++] = verts[0];
			}
			total++;
			verts++;
			order+=2;
		}
	}

	dr_ConPrintf("Total: %i,  Listed: %i\n",total,size);
	last = -1;
	while (total>0) {
		int lowest = 999999;
		int index;
		for (count = 0; count < size; count++) {
			if (countl[count]==0) continue;
			if (lowest > (list+count)->v[0]) {
				lowest = (list+count)->v[0];
				index = count;
			}
		}

		if (last<0) last = lowest;
		if (lowest!=last) {last=lowest;dr_ConPrintf("\n");}
		dr_ConPrintf("%i - ",countl[index]);printVertU((list+index)->v);printDist((list+index)->v);
		countl[index] = 0; total--;
	}
}
*/

#ifndef _WIN32
void print_edge_table(entry_t **table, aliashdr_t *paliashdr, int posenum){
	trivertx_t	*verts;
	int		*order;
	int		count;
	int		strip;
	float *vert1;
	float *vert2;
	float *vert3;

	dr_ConPrintf("Printing Edge Table... \n");
	verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);

	vert1 = malloc(sizeof(float)*3);
	vert2 = malloc(sizeof(float)*3);
	vert3 = malloc(sizeof(float)*3);
	while (1)
	{
		float *norm;
		edge_t *e;
		// get the vertex count and primitive type
		strip = 1;
		count = *order++;
		if (!count) break;		// done

		if (count < 0) {
			count = -count;
			strip = 0;
		}
		vert1[0] = (float) verts->v[0];
		vert1[1] = (float) verts->v[1];
		vert1[2] = (float) verts->v[2];
		verts++;

		vert2[0] = (float) verts->v[0];
		vert2[1] = (float) verts->v[1];
		vert2[2] = (float) verts->v[2];
		verts++;

		vert3[0] = (float) verts->v[0];
		vert3[1] = (float) verts->v[1];
		vert3[2] = (float) verts->v[2];

		order+=6;
		count-=3;

		printEdge(get(table, vert1, vert2));
		//printVert(vert1);
		//printVert(vert2);
		do
		{
			//printVert(vert3);
			printEdge(get(table, vert2, vert3));
			printEdge(get(table, vert3, vert1));

			verts++;
			if (count<=0) break; // done with strip/fan

			if (strip) {
				vert1[0] = vert2[0];
				vert1[1] = vert2[1];
				vert1[2] = vert2[2];
			}
			vert2[0] = vert3[0];
			vert2[1] = vert3[1];
			vert2[2] = vert3[2];

			order += 2;
			vert3[0] = (float) verts->v[0];
			vert3[1] = (float) verts->v[1];
			vert3[2] = (float) verts->v[2];
		} while (count--); // end while spinning through a strip/fan
	} // end while more strips/fans exist
} // end if neighbouring information needs building
#endif

void findNearestVertex(aliashdr_t *paliashdr, int posenum, float *vert1, float *vert2, float *dist) {

	trivertx_t  *verts;
	float  d1 = 999999;
	float d2 = 999999;
	float t1;
	float t2;
	float v1[3],v2[3];
	int  *order;
	int  count;
	int  strip;

	verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);
	count = *order++;

	while (count != 0)
	{
		if (count < 0)
		{
			count = -count;
		}
#ifndef _WIN32
		t1 = sqrt(powf(verts->v[0]-vert1[0],2) + powf(verts->v[1]-vert1[1],2) + powf(verts->v[2]-vert1[2],2));
		t2 = sqrt(powf(verts->v[0]-vert2[0],2) + powf(verts->v[1]-vert2[1],2) + powf(verts->v[2]-vert2[2],2));
#else
		t1 = sqrt(pow(verts->v[0]-vert1[0],2) + pow(verts->v[1]-vert1[1],2) + pow(verts->v[2]-vert1[2],2));
		t2 = sqrt(pow(verts->v[0]-vert2[0],2) + pow(verts->v[1]-vert2[1],2) + pow(verts->v[2]-vert2[2],2));
#endif


		if (t1>0 && t1<d1) {
			d1 = t1;
			v1[0] = verts->v[0];
			v1[1] = verts->v[1];
			v1[2] = verts->v[2];
		}
		if (t2>0 && t2<d2) {
			d2 = t2;
			v2[0] = verts->v[0];
			v2[1] = verts->v[1];
			v2[2] = verts->v[2];
		}

		if (t1>0 && t1<*dist) {
			*dist = t1;
			v1[0] = verts->v[0];
			v1[1] = verts->v[1];
			v1[2] = verts->v[2];
		}
		if (t2>0 && t2<*dist) {
			*dist = t2;
			v2[0] = verts->v[0];
			v2[1] = verts->v[1];
			v2[2] = verts->v[2];
		}
		verts++;
		order+=2; // add 2 per vertex
		count--;
	}

//	dr_ConPrintf("\nVertex: ");printVert(vert1);
//	dr_ConPrintf(" best matches: ");printVert(v1);
//
//	dr_ConPrintf("\nVertex: ");printVert(vert2);
//	dr_ConPrintf(" best matches: ");printVert(v2);
}

#ifndef _WIN32
void checkForSilCrease (edge_t *e, float *cameravec,
		int *creaseSize, float **creaseVecList,
		int *silSize, float **silVecList,
		int *oddSize, float **oddVecList)
{
	//float *vert1, float *vert2, float *vert3, float *eyeT ) {
	int size = 0;
	float eyeVec[3], pt[3];
	float cosEyeN1, cosEyeN2;
	float edgeCenter[3];
	if(e->n1==NULL)
		return;
	else if (e->n2==NULL)// && (e->rn & 1) != 1)
	{
		/*if ((e->rn & 1) != 1) {// dr_ConPrintf("FINAL EDGE!!!!!\n");
			//else dr_ConPrintf("NOTNOT FINAL EDGE!!!!!\n");
			size = *oddSize;
			*oddVecList = realloc(*oddVecList, sizeof(float)*(size+6));
			(*oddVecList)[size++] = e->v1[0];
			(*oddVecList)[size++] = e->v1[1];
			(*oddVecList)[size++] = e->v1[2];
			(*oddVecList)[size++] = e->v2[0];
			(*oddVecList)[size++] = e->v2[1];
			(*oddVecList)[size++] = e->v2[2];
			*oddSize = size;
		} else {
		*/
			size = *oddSize;
			*oddVecList = realloc(*oddVecList, sizeof(float)*(size+6));
			(*oddVecList)[size++] = e->v1[0];
			(*oddVecList)[size++] = e->v1[1];
			(*oddVecList)[size++] = e->v1[2];
			(*oddVecList)[size++] = e->v2[0];
			(*oddVecList)[size++] = e->v2[1];
			(*oddVecList)[size++] = e->v2[2];
			*oddSize = size;
		//}
		return;
	}
	//dr_ConPrintf("\tChecking for Sil/Crease on edge\n");
	//printEdge(e);
	//if (assertEdge(e)) abort();

	// get center of edge.
		// seems better than center of surface - WE WIN!

	/*VectorAdd(e->v1, e->v2, edgeCenter);
	edgeCenter[0] /= 2;
	edgeCenter[1] /= 2;
	edgeCenter[2] /= 2;
	*/

	/*
	// get eye vector to edge center
	VectorSubtract(eyePosition,edgeCenter, eyeVec);
	//dr_ConPrintf("Unnormal Eye Vec: ");printVert(eyeVec);
	vecNormalize(eyeVec);

	eyePosition[0] = eyeVec[0] * 1000;
	eyePosition[1] = eyeVec[1] * 1000;
	eyePosition[2] = eyeVec[2] * 1000;
	*/
	//dr_VectorScale(cameravec,1000.0,cameravec);
	//dr_VectorScale(cameravec,-1.0,cameravec);
	//VectorAdd(cameravec,edgeCenter,pt);
	//dr_ConPrintf("Eye Vector:    ");printVert(eyeVec);
	//glPushAttrib(GL_ALL_ATTRIB_BITS);
	//glLineWidth(20);
	// DEBUG: draw lines from edge center pointing at eye position.
	/*
	glColor3f(1.0,0.0,0.0);
	glBegin(GL_LINES);
		glVertex3f(0.0,0.0,0.0);
		glColor3f(0.0,0.0,1.0);
		glVertex3fv(pt);
	glEnd();
	glPopAttrib();
	*/

	// FIXME: do this in calling function instead
	//vecNormalize(cameravec);
	// get cosines of angles between eye vector and normals 1 and 2
	//cosEyeN1 = DotProduct(eyeVec,e->n1);
	//cosEyeN1 = DotProduct(cameravec,e->n1);
//	if (e->n2!=NULL) {
		//cosEyeN2 = DotProduct(eyeVec,e->n2);
//		cosEyeN1 = DotProduct(e->n1,e->n2);
		//dr_ConPrintf("Eye Vector: ");printVert(eyeVec);
		// if surface 1 is backfacing or orthogonal to eye vector
//		if (cosEyeN1 < 0) {
			// if surface 2 is backfacing or orthogonal to eye vector
//			if (cosEyeN2 < 0)
//			{
//				return; // don't paint anything
//			}
		// both are front facing
//		} else if (cosEyeN2 >= 0) {

/*
			GLfloat cosN1N2 = abs(vecDotProduct(e->n1, e->n2));
			float rads = acosf((float)cosN1N2);
			float threshold = tech_ridge_thresh.value*3.141592654/180;
			if (rads<threshold) {
				size = *creaseSize;
				*creaseVecList = realloc(*creaseVecList, sizeof(float)*(size+6));
				(*creaseVecList)[size++] = e->v1[0];
				(*creaseVecList)[size++] = e->v1[1];
				(*creaseVecList)[size++] = e->v1[2];
				(*creaseVecList)[size++] = e->v2[0];
				(*creaseVecList)[size++] = e->v2[1];
				(*creaseVecList)[size++] = e->v2[2];
				*creaseSize = size;
			}
			return;
*/

//		}
//	}else if (!(e->rn & 1))
//		return;

//	GLfloat cosN1N2 = vecDotProduct(e->n1, e->n2);
//	float threshold = 6*M_PI/4;
//	if(acosf((float)cosN1N2) > threshold)
//		return;

	// surface 2 is backfacing and surface 1 is NOT or vice-versa
	// silhouette!
//	size = *silSize;
//	*silVecList = realloc(*silVecList, sizeof(float)*(size+6));
//	(*silVecList)[size++] = e->v1[0];
//	(*silVecList)[size++] = e->v1[1];
//	(*silVecList)[size++] = e->v1[2];
//	(*silVecList)[size++] = e->v2[0];
//	(*silVecList)[size++] = e->v2[1];
//	(*silVecList)[size++] = e->v2[2];
//	*silSize = size;
}

void DrawSilsCreases (aliashdr_t *paliashdr, entry_t **table, int posenum)
{
	trivertx_t  *verts;
	edge_t  *e;
	float mdlMatrix[16];
	float invMatrix[16];
	float tmpMatrix[16];
	float oldMatrix[16];
	float worldcampos[] = {0,0,0,1};
	float worldcamdir[] = {0,0,1,1};
	float cameravec[] = {0,0,0,0};
	float camerapos[] = {0,0,0,0};
	float cameradir[] = {0,0,0,0};
	float tempvec[] = {0,0,0,0};
	float vert1[4];
	float vert2[4] = {0,0,0,0};
	float vert3[3];
	// set to null so they can be realloc'd later
	float *silVecList = NULL;
	float *creaseVecList = NULL;
	float *oddVecList = NULL;
	int  *order;
	int  count;
	int oldCount;
	int  strip;
	int  cSize;
	int  sSize;
	int  oSize;
	float distToMdl;
	cSize = 0;
	sSize = 0;
	oSize = 0;


	// MODELVIEW matrix: translation from object to world space
	glGetFloatv(GL_MODELVIEW_MATRIX, mdlMatrix);

	// draw the object origin in green
//	glPushAttrib(GL_ALL_ATTRIB_BITS);
/*	glPointSize(12);
	glColor3f(0,1,0);
	glBegin(GL_POINTS);
		glVertex3f(mdlMatrix[12],mdlMatrix[13],mdlMatrix[14]);
	glEnd();
*/
	// the inverse should be the translation from world to object space
	matPrint(mdlMatrix,"before:");
	Invert2(mdlMatrix,invMatrix);
	//invertMatrixOld(mdlMatrix);
	//matPrint(invMatrix,"After:");
	//glMultMatrixf(invMatrix);
	//glGetFloatv(GL_MODELVIEW_MATRIX, mdlMatrix);
	//matPrint(mdlMatrix,"ID:");

	//invertMatrix(tmpMatrix, 4);
	// The world-origin is the camera position in GL. Translate to object space.
	//mat4Mult(mdlMatrix,tmpMatrix,oldMatrix);
//	camerapos[0] = invMatrix[12];
//	camerapos[1] = invMatrix[13];
//	camerapos[2] = invMatrix[14];
/*
	idvec[0] = paliashdr->eyeposition[0];
	idvec[1] = paliashdr->eyeposition[1];
	idvec[2] = paliashdr->eyeposition[2];
	idvec[3] = 1.0f;
	vectorMatrixMult(worldcampos,invMatrix,camerapos);
	vectorMatrixMult(worldcamdir,invMatrix,cameradir);
	VectorSubtract(cameradir, camerapos, cameravec);
	vecNormalize(cameravec);
	VectorAdd(cameradir, tempvec, tempvec);
	dr_VectorScale(tempvec, 1000.0, tempvec);
	VectorAdd(camerapos,tempvec,tempvec);
*/

/*
	glPointSize(32);
	glColor3f(1.0,0.0,0.0);
	glBegin(GL_POINTS);
	glVertex3fv(camerapos);
	glEnd();

	glLineWidth(40);
	glBegin(GL_LINES);

	glColor3f(1.0,1.0,0.0);
	glVertex3fv(camerapos);
	glColor3f(0.0,0.0,1.0);
	glVertex3fv(cameradir);

	glColor3f(1.0,0.0,1.0);
	glVertex3fv(camerapos);
	glColor3f(0.0,1.0,1.0);
	glVertex3fv(tempvec);
	glEnd();
	glPopAttrib();
*/
	//mat4Invert(mdlMatrix, invMatrix);
	//mat4vec3Mult(mdlMatrix,idvec,camera);
	//mat4Mult(tmpMatrix,mdlMatrix,tmpMatrix);
//	dr_ConPrintf("Eye Camera:    ");printVert(camerapos);

	glPushAttrib(GL_ALL_ATTRIB_BITS);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE);
	glDepthMask( GL_TRUE );
	glDisable(GL_BLEND);
	glColor3f(0,1,0);
	if (tech_antialias.value)
		glEnable(GL_LINE_SMOOTH);
	else
		glDisable(GL_LINE_SMOOTH);

	verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);

	glLineWidth(10);
	glColor3f(0,1,0);
	vert1[0] = (float)verts->v[0];
	vert1[1] = (float)verts->v[1];
	vert1[2] = (float)verts->v[2];
	vert1[3] = 1.0f;
	vectorMatrixMult(vert1,mdlMatrix,vert2);
	glBegin(GL_LINES);
	glVertex3fv(gun_pos);
	glVertex3fv(vert2);
	glEnd();

	count = *order++;
	//glBegin(GL_LINES);
	while (count != 0)
	{
		strip = 1;
		if (count < 0)
		{
			count = -count;
			strip = 0;
		}
		vert1[0] = (float)verts->v[0];
		vert1[1] = (float)verts->v[1];
		vert1[2] = (float)verts->v[2];
		verts++;
		vert2[0] = (float)verts->v[0];
		vert2[1] = (float)verts->v[1];
		vert2[2] = (float)verts->v[2];
		verts++;
		order+=4; // add 2 per vertex
		count-=2; // decrement once per vertex

		// check edge vert1-vert2
		e = get(table, vert1, vert2);
		//if (e->n2 == NULL && ((e->rn & 1) == 0))
			//findNearestVertex(paliashdr, posenum, vert1, vert2, &dist);
		//if (!assertEdge(e))
		checkForSilCrease(e, cameravec,
			&cSize, &creaseVecList, &sSize, &silVecList, &oSize, &oddVecList);
		while (count>0) {

			vert3[0] = (float)verts->v[0];
			vert3[1] = (float)verts->v[1];
			vert3[2] = (float)verts->v[2];
			verts++;
			order+=2; // add 2 per vertex
			count--; // decrement once per vertex
			/*
			glVertex3fv(vert1);
			glVertex3fv(vert2);
			glVertex3fv(vert2);
			glVertex3fv(vert3);
			glVertex3fv(vert1);
			glVertex3fv(vert3);
			*/
			// check edge vert2-vert3
			e = get(table, vert2, vert3);
			//if (e->n2 == NULL && ((e->rn & 1) == 0))
				//findNearestVertex(paliashdr, posenum, vert2, vert3, &dist);

			//if (!assertEdge(e))
			checkForSilCrease(e, cameravec,
				&cSize, &creaseVecList, &sSize, &silVecList, &oSize, &oddVecList);

			// check edge vert3-vert1
			e = get(table, vert3, vert1);
			//if (e->n2 == NULL && ((e->rn & 1) == 0))
				//findNearestVertex(paliashdr, posenum, vert3, vert1, &dist);

			//if (!assertEdge(e))
				checkForSilCrease(e, camerapos,
				&cSize, &creaseVecList, &sSize, &silVecList, &oSize, &oddVecList);

			if (strip) {
				vert1[0] = vert2[0];
				vert1[1] = vert2[1];
				vert1[2] = vert2[2];
				vert2[0] = vert3[0];
				vert2[1] = vert3[1];
				vert2[2] = vert3[2];
			} else {
				vert2[0] = vert3[0];
				vert2[1] = vert3[1];
				vert2[2] = vert3[2];
			}
		}
		count = *order++;
	}

	//glEnd();
//	glPopAttrib();

//	glPushAttrib(GL_ALL_ATTRIB_BITS);
//	glDisable(GL_TEXTURE_2D);
//	glDisable(GL_CULL_FACE);
//	glDepthMask( GL_TRUE );
//	glDisable(GL_BLEND);
	if (tech_antialias.value)
		glEnable(GL_LINE_SMOOTH);
	else
		glDisable(GL_LINE_SMOOTH);

	#ifdef GL_EXT_vertex_array
	{
		glEnable(GL_VERTEX_ARRAY_EXT);
		if (tech_crease_mode.value && cSize>0) {
			setColor(tech_crease_color.value);
			glLineWidth(tech_crease_width.value);
			cSize /= 3;
			glVertexPointerEXT(3, GL_FLOAT, 0, cSize, creaseVecList);
			glDrawArraysEXT(GL_LINES, 0, cSize);
		}
		if (tech_sil_mode.value && sSize>0) {
			setColor(tech_sil_color.value);
			glLineWidth(tech_sil_width.value);
			sSize /= 3;
			glVertexPointerEXT(3, GL_FLOAT, 0, sSize, silVecList);
			glDrawArraysEXT(GL_LINES, 0, sSize);
		}

		if (oSize>0) {
			glColor3f(1,0,0);
			glLineWidth(tech_line_width.value);
			oSize /= 3;
			glVertexPointerEXT(3, GL_FLOAT, 0, oSize, oddVecList);
			glDrawArraysEXT(GL_LINES, 0, oSize);
		}
	}
	#else
	{
		int index = 0;
		if (tech_crease_mode.value && cSize>0) {
			setColor(tech_crease_color.value);
			glLineWidth(tech_crease_width.value);
			glBegin(GL_LINES);
			while (cSize>index) {
				glVertex3fv(creaseVecList+index);
				index+=3;
			}
			glEnd();
		}
		if (tech_sil_mode.value && sSize>0) {
			setColor(tech_sil_color.value);
			glLineWidth(tech_sil_width.value);
			glBegin(GL_LINES);
			index = 0;
			while (sSize>index) {
				glVertex3fv(silVecList+index);
				index+=3;
			}
			glEnd();
		}
	}
	#endif

	glPopAttrib();
	// free memory
	free(silVecList);
	free(creaseVecList);
}

void build_edge_table(entry_t **table, char *name, aliashdr_t *paliashdr, int posenum){
	float		first[3];
	trivertx_t	*verts;
	int		*order;
	int		count;
	int		strip;

	//dr_ConPrintf("Building new edge hash... %s\t%i\n", name, posenum);
	verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);

	while (1)
	{
		float *norm;
		edge_t *e;
		float *vert1;
		float *vert2;
		float *vert3;

		// get the vertex count and primitive type
		strip = 1;
		count = *order++;
		if (!count) break;		// done

		if (count < 0) {
			count = -count;
			strip = 0;
		}
		vert1 = malloc(sizeof(float)*3);
		vert2 = malloc(sizeof(float)*3);
		vert3 = malloc(sizeof(float)*3);
		*vert1 = (float) verts->v[0];
		*(vert1+1) = (float) verts->v[1];
		*(vert1+2) = (float) verts->v[2];
		verts++;
		*vert2 = (float) verts->v[0];
		*(vert2+1) = (float) verts->v[1];
		*(vert2+2) = (float) verts->v[2];
		verts++;

		*vert3 = (float) verts->v[0];
		*(vert3+1) = (float) verts->v[1];
		*(vert3+2) = (float) verts->v[2];

		order+=6;
		count-=3;

		e = get(table, vert1, vert2);
		if (e == NULL) {
			e = calloc(1, sizeof(edge_t));
			e->v1 = vert1;
			e->v2 = vert2;
			add(table,e);

		}
		norm = malloc(sizeof(float)*3);
		triNorm(verts->lightnormalindex,vert1,vert2,vert3,norm);
		if(isNaNm(norm)) {
			e->rn |= 4;
		} else
			addnorm(e,norm);
		do
		{
			if (norm==NULL) {
				norm = malloc(sizeof(float)*3);
				triNorm(verts->lightnormalindex,vert1,vert2,vert3,norm);
			}
			e = get(table, vert2, vert3);
			if (e == NULL) {
				e = calloc(1, sizeof(edge_t));
				e->v1 = vert2;
				e->v2 = vert3;
				add(table,e);
			}
			if(isNaNm(norm)) {
				//printVert(vert1);printVert(vert2);printVert(vert3);dr_ConPrintf("HEREHEREHERE: \n\n\n");
				e->rn |= 4;
			} else
				addnorm(e,norm);

			e = get(table, vert3, vert1);
			if (e == NULL) {
				e = calloc(1, sizeof(edge_t));
				e->v1 = vert3;
				e->v2 = vert1;
				add(table, e);
			}
			if(isNaNm(norm))
				e->rn |= 4;
			else
				addnorm(e,norm);

			norm = NULL;
			verts++;
			if (count<=0) {
				break; // done with strip/fan
			}

			if (strip)
				vert1 = vert2;

			vert2 = vert3;

			order += 2;
			vert3 = malloc(sizeof(float)*3);
			*vert3 = (float) verts->v[0];
			*(vert3+1) = (float) verts->v[1];
			*(vert3+2) = (float) verts->v[2];
		} while (count--); // end while spinning through a strip/fan
	} // end while more strips/fans exist
} // end if neighbouring information needs building
#endif


//draw one animated object
void DrawSilhouettes(aliashdr_t *paliashdr, int posenum, float * shadedots,
			float shadelight, float ratio, float *vclist, entity_t * currententity )
{
	float 	l;
	trivertx_t	*verts;
	int		*order;
	int		count, numverts = 0;

	verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);

	while (1)
	{
		// get the vertex count and primitive type
		count = *order++;
		if (!count) break; // no more strips/fans
		if (count < 0) {
			count = -count;
			glBegin (GL_TRIANGLE_FAN);
		} else
			glBegin (GL_TRIANGLE_STRIP);

		do
		{
			if (tech_sil_mode.value==SIL_SHADED) // shaded (mode 2)
			{
				float r,g,b, color[3];
				int c = (int)tech_sil_color.value;
				b = (c % 256)/256.0;
				c /= 256;
				g = (c % 256)/256.0;
				r = (c / 256)/256.0;
				memcpy(color,vclist + numverts*3, sizeof(float)*3);

				// FIX THIS RATIO MESS!
				color[0] = r + ((color[0] - b)*ratio);
				color[1] = g + ((color[1] - g)*ratio);
				color[2] = b + ((color[2] - b)*ratio);

				glColor3fv(color);
			}
			order += 2;
			glVertex3f((float)verts->v[0], (float)verts->v[1], (float)verts->v[2]);
			verts++;
			numverts++;
		} while (--count);
		glEnd ();
	}
}

/*
=============
GL_DrawAliasFrame
=============
*/
extern	vec3_t			lightspot;
EXPORT void GL_DrawAliasFrame (aliashdr_t *paliashdr, int posenum,
						float *shadedots, float shadelight,
						entity_t * currententity)
{
	char *name = currententity->model->name;
	//entry_t **table;
	trivertx_t *verts;
	int *order;
	int count;
	int index;
	int numverts = 0;
	const float alpha = 0.76, p = 0.78;
	float distance = 0.0f, sratio, cratio;
	float *vclist=NULL;
	float light[3];
	float color[4];
	float cool[3];
	float warm[3];

	if(tech_shade_model_mode.value){
		getColor(tech_shade_warm.value, warm);
		getColor(tech_shade_cool.value, cool);
	}

//	cool[0] = 0.0f; cool[1] = 0.0f; cool[2] = 0.4f;
//	warm[0] = 1.0f; warm[1] = 1.0f; warm[2] = 0.0f;
	// build the edge table if we haven't seen this model and pose (frame) before

	/* No Use building the edge table - it's garbage anyway
	 *
	if (get_table(name, posenum, &table)==0)
	{
		build_edge_table(table, name, paliashdr, posenum);
		// print edge table - for debugging.
		// print_edge_table(table, paliashdr,posenum);
	}
	 *
	 */

	glPushAttrib(GL_ALL_ATTRIB_BITS);
	if (tech_antialias.value)
		glEnable(GL_LINE_SMOOTH);
	else
		glDisable(GL_LINE_SMOOTH);

	//dr_ConPrintf( "%s\t%i\n", name, posenum );

	// Textures don't work as they should
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_TEXTURE_1D);
	/*
	glCullFace(GL_FRONT);
	glEnable(GL_CULL_FACE);
	glDepthMask(GL_TRUE);
	glDepthFunc(GL_LEQUAL);
	glCullFace(GL_BACK);
	glDepthMask(GL_FALSE);	// depth buffer is read only
	glDepthFunc(GL_LEQUAL);
	*/
	glPolygonMode(GL_FRONT, GL_FILL);

	verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
	verts += posenum * paliashdr->poseverts;
	order = (int *)((byte *)paliashdr + paliashdr->commands);

	cratio = 1.0f;
	sratio = 0.0f;

	if (tech_crease_depth.value || tech_sil_depth.value) {
		float mdlMatrix[16], mdl_pos[4], v[4];
		glGetFloatv(GL_MODELVIEW_MATRIX, mdlMatrix);
		v[0] = (float)verts->v[0];
		v[1] = (float)verts->v[1];
		v[2] = (float)verts->v[2];
		v[3] = 1.0f;

		// it this is the gun, use it for distance
		if (strncmp(name+6,"v_",2)==0)
		{
			gun_pos[0] = 0.0f; 	gun_pos[1] = 0.0f;
			gun_pos[2] = 0.0f; 	gun_pos[3] = 0.0f;
			vectorMatrixMult(v,mdlMatrix,gun_pos);
		} else {
			float distance, dist[3];
			vectorMatrixMult(v,mdlMatrix,mdl_pos);
			VectorSubtract(mdl_pos,gun_pos,dist);
			distance = sqrt(dist[0]*dist[0] + dist[1]*dist[1] + dist[2]*dist[2]);

			if (tech_crease_depth.value) {
				cratio = distance / (float)tech_crease_depth.value;
				if (cratio>1.0f) cratio = 0.0f;
				else cratio = 1 - cratio;
			}
			if (tech_sil_depth.value) {
				sratio = distance / (float)tech_sil_depth.value;
				if (sratio>1.0f) sratio = 1.0f;
			}
		}
	}

	while (1){
		// get the vertex count and primitive type
		count = *order++;
		if (!count) break; // count out, we're done
		if (count < 0){
			count = -count;
			glBegin (GL_TRIANGLE_FAN);
		}else
			glBegin (GL_TRIANGLE_STRIP);

		vclist = realloc(vclist,sizeof(float)*3*(numverts+count));
		do{
			float d,l, c = 1.0f;
			float temp[3];
			// do we want to use cool-to-warm Gooch shading?
			if(tech_shade_model.value){
				color[0] = color[1] = color[2] = 1.0f;

				// this is crap, linear blend please!
				// better yet, blend with tex color!
				if (tech_color_models.value) {
					temp[0] = color[0]/2;
					temp[1] = color[1]/2;
					temp[2] = color[2]/2;
					color[0] += temp[0];
					color[1] += temp[1];
					color[2] += temp[2];
				}
				// texture coordinates come from the draw list
				//glTexCoord2f (((float *)order)[0], ((float *)order)[1]);
				//glTexCoord1f (l);
				color[4] = 1.0;
				/*
			else if (ainpr_sil_mode.value==SIL_TOON)//textured
			{
				// normals and vertexes come from the frame list
				l = shadedots[verts->lightnormalindex] * shadelight;
			}*/

				l = shadedots[verts->lightnormalindex] * shadelight;
				// splashback enabled with mode 2
				if(tech_shade_model_mode.value == 2)
#ifndef _WIN32
					c = powf(alpha*fabs(l)+ (1 - alpha),p);
#else
					c = pow(alpha*fabs(l)+ (1 - alpha),p);
#endif


				color[0] *= (((1 + l) / 2) *warm[0] + (1-(1+l)/2)*cool[0]) *c;
				color[1] *= (((1 + l) / 2) *warm[1] + (1-(1+l)/2)*cool[1]) *c;
				color[2] *= (((1 + l) / 2) *warm[2] + (1-(1+l)/2)*cool[2]) *c;

				glColor4fv(color);
			}else
				glColor3f(0.0f,0.0f,0.5f);

			// copy color to memory
			memcpy((void *)(vclist + numverts*3),(void *)color,sizeof(float)*3);
			// added float casts
			/* Draw stuff here */
			glVertex3f ((float)verts->v[0], (float)verts->v[1], (float)verts->v[2]);
			numverts++;
			order+=2;
			verts++;
		} while (--count);

		glEnd ();
	}

	if (tech_crease_mode.value) {
		int fan;
		float colorAdd, vcol1[3], vcol2[3], vcol3[3], vert1[3], vert2[3], vert3[3];
		numverts=0;

		if (tech_crease_depth.value)
			colorAdd = ((float)tech_color_thresh.value)*cratio;
		else colorAdd = (float)tech_color_thresh.value;

		verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
		verts += posenum * paliashdr->poseverts;
		order = (int *)((byte *)paliashdr + paliashdr->commands);
		glLineWidth(tech_crease_width.value);
		while (1){
			// get the vertex count and primitive type
			count = *order++;
			if (!count) break; // count out, we're done
			if (count < 0) {
				fan = 1;
				count = -count;
			} else fan = 0;

			vert1[0] = (float)verts->v[0]; vert1[1] = (float)verts->v[1]; vert1[2] = (float)verts->v[2];
			memcpy(vcol1, (void *)(vclist + numverts*3), sizeof(float)*3);
			verts++;
			numverts++;
			order+=2;
			vert2[0] = (float)verts->v[0]; vert2[1] = (float)verts->v[1]; vert2[2] = (float)verts->v[2];
			memcpy(vcol2, (void *)(vclist + numverts*3), sizeof(float)*3);
			numverts++;
			order+=2;
			verts++;
			count-=2;
			// add a constant value to each R, G, B for showing edges
			vcol1[0] += colorAdd; vcol2[0] += colorAdd;
			vcol1[1] += colorAdd; vcol2[1] += colorAdd;
			vcol1[2] += colorAdd; vcol2[2] += colorAdd;

			glBegin(GL_LINES);

			do {
				vert3[0] = (float)verts->v[0]; vert3[1] = (float)verts->v[1]; vert3[2] = (float)verts->v[2];
				memcpy(vcol3, (void *)(vclist + numverts*3), sizeof(float)*3);
				numverts++;
				order+=2;
				verts++;

				vcol3[0] += colorAdd;
				vcol3[1] += colorAdd;
				vcol3[2] += colorAdd;

				// line from 1 to 2
				glColor3fv(vcol1);
				glVertex3fv(vert1);
				glColor3fv(vcol2);
				glVertex3fv(vert2);

				// line from 2 to 3
				glVertex3fv(vert2);
				glColor3fv(vcol3);
				glVertex3fv(vert3);

				// line from 3 to 1
				glVertex3fv(vert3);
				glColor3fv(vcol1);
				glVertex3fv(vert1);

				if (fan==0) {
					memcpy(vert1, vert2, sizeof(float)*3);
					memcpy(vcol1, vcol2, sizeof(float)*3);
				}

				memcpy(vert2, vert3, sizeof(float)*3);
				memcpy(vcol2, vcol3, sizeof(float)*3);
			} while (--count);
			glEnd ();
		}
	} // end creases

	if (tech_sil_mode.value)
	{
		if (tech_sil_mode.value==SIL_FLAT) {
			glShadeModel(GL_FLAT);
			setColor(tech_sil_color.value);
		} else glShadeModel(GL_SMOOTH);
		glPolygonMode(GL_FRONT, GL_LINE);//for some reason the front polygons are backward ;)
		glDepthFunc(GL_LEQUAL);
		glCullFace(GL_BACK);
		glDepthMask(GL_FALSE);
    		glLineWidth(tech_sil_width.value);
		DrawSilhouettes(paliashdr, posenum, shadedots, shadelight, sratio, vclist, currententity);
	}
/*
 * CRAP way of doing creases which CLEARLY doesn't work, despite hours and hours of work.
 *
	if (tech_crease_mode.value || tech_sil_mode.value) {
		DrawSilsCreases (paliashdr, table, posenum);
	}
*/

	glPopAttrib();
	free(vclist);
}

/*
=============
GL_DrawAliasShadow
=============
Only gets called if the cvar r_shadows is set.
*/
EXPORT void GL_DrawAliasShadow (aliashdr_t *paliashdr, entity_t * currententity,
						 int posenum, vec3_t shadevector,
						 vec3_t lightspot)
{
	trivertx_t	*verts;
	int		*order;
	vec3_t	point;
	float	height, lheight, offset;
	int		count, i;
	float light[3];
	float origcolor[4];
	int times = 1;
	int shadowable = 1;
	if(tech_shadows.value == 1)
		times = 3;
	else if (tech_shadows.value == 2)
		times = 1;
	if( !strncmp( &(currententity->model->name[6]), "wiz", 3) )
		shadowable = 0;
	else if( !strncmp( &(currententity->model->name[6]), "gib", 3) )
		shadowable = 0;
	else if( !strncmp( &(currententity->model->name[6]), "ger", 3) )
		shadowable = 0;
	else if( !strncmp( &(currententity->model->name[6]), "fla", 3) )
		shadowable = 0;
	else if( !strncmp( &(currententity->model->name[6]), "v_", 2) )
		shadowable = 0;
	else if( !strncmp( &(currententity->model->name[6]), "g_", 2) )
		shadowable = 0;
	if(shadowable != 1) return;

	glGetFloatv(GL_CURRENT_COLOR,(GLfloat *)origcolor);

	offset = tech_shadow_offset.value;

	lheight = currententity->origin[2] - lightspot[2];
	height = -lheight + 1.0;

	for(i = 1; i < times+1; i++){
		verts = (trivertx_t *)((byte *)paliashdr + paliashdr->posedata);
		verts += posenum * paliashdr->poseverts;
		order = (int *)((byte *)paliashdr + paliashdr->commands);

		while (1)
		{
			// get the vertex count and primitive type
			count = *order++;
			if (!count)
				break;		// done
			if (count < 0)
			{
				count = -count;
				glBegin (GL_TRIANGLE_FAN);
			}
			else
				glBegin (GL_TRIANGLE_STRIP);

			do
			{
				float c[3];
				// texture coordinates come from the draw list
				// (skipped for shadows) glTexCoord2fv ((float *)order);
				order += 2;

				if(tech_shadows.value){
					getColor(tech_shadow_color.value,c);
					// do hacky shit to move the shadow a bit
					if(i == 3 || times == 1){
						float scale = tech_shadow_umbra.value; // scale middle one down
						point[0] = verts->v[0] * paliashdr->scale[0] + paliashdr->scale_origin[0];
						point[1] = verts->v[1] * paliashdr->scale[1] + paliashdr->scale_origin[1];
						point[2] = verts->v[2] * paliashdr->scale[2] + paliashdr->scale_origin[2];
						point[0] -= shadevector[0]*(point[2]+lheight)
								* tech_shadow_scale.value*scale;
						point[1] -= shadevector[1]*(point[2]+lheight)
								* tech_shadow_scale.value*scale;
						glColor4f(c[0],c[1],c[2],tech_shadow_alpha.value);
					}else{
						glColor4f(c[0],c[1],c[2],tech_shadow_alpha.value*0.47f);
						point[0] = verts->v[0] * paliashdr->scale[0] + paliashdr->scale_origin[0];
						point[1] = verts->v[1] * paliashdr->scale[1] + paliashdr->scale_origin[1];
						point[2] = verts->v[2] * paliashdr->scale[2] + paliashdr->scale_origin[2];
						point[0] -= shadevector[0]*(point[2]+lheight)
								* tech_shadow_scale.value;
						point[1] -= shadevector[1]*(point[2]+lheight)
								* tech_shadow_scale.value + offset;
					}
				}else{
					point[0] = verts->v[0] * paliashdr->scale[0] + paliashdr->scale_origin[0];
					point[1] = verts->v[1] * paliashdr->scale[1] + paliashdr->scale_origin[1];
					//doesn't make sense
					//point[2] = verts->v[2] * paliashdr->scale[2] + paliashdr->scale_origin[2];
				}
				point[2] = height;

				glVertex3fv (point);
				verts++;
			} while (--count);
			glEnd ();
		}
		offset = -offset;
		height += 0.015;
		glColor4fv((GLfloat *)origcolor);
	}
}


/*
============
R_PolyBlend
============
*/
EXPORT void R_PolyBlend (float * v_blend, cvar_t * gl_polyblend)
{
	if (!gl_polyblend->value)
		return;
	if (!v_blend[3])
		return;

	dr_GL_DisableMultitexture();

	glDisable (GL_ALPHA_TEST);
	glEnable (GL_BLEND);
	glDisable (GL_DEPTH_TEST);
	glDisable (GL_TEXTURE_2D);

	glLoadIdentity ();

	glRotatef (-90,  1, 0, 0);		// put Z going up
	glRotatef (90,  0, 0, 1);		// put Z going up

	glColor4fv (v_blend);

	glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);
	glEnable (GL_ALPHA_TEST);
}

/*
================
R_DrawSequentialPoly

Systems that have fast state and texture changes can
just do everything as it passes with no need to sort
================
*/
EXPORT void R_DrawSequentialPoly (msurface_t *s, int lightmap_textures,
						   qboolean * lightmap_modified,
						   glRect_t * lightmap_rectchange, byte * lightmaps,
						   int lightmap_bytes, int solidskytexture,
						   float * speedscale, int alphaskytexture,
						   qboolean gl_mtexable, int currenttexture,
						   int * cnttextures, double realtime,
						   lpMTexFUNC qglMTexCoord2fSGIS, int gl_lightmap_format,
						   lpSelTexFUNC qglSelectTextureSGIS, vec3_t r_origin )
{
	glpoly_t	*p;
	float		*v;
	int			i;
	texture_t	*t;
	vec3_t		nv;
	glRect_t	*theRect;

	//
	// normal lightmapped poly
	//

	if (! (s->flags & (SURF_DRAWSKY|SURF_DRAWTURB|SURF_UNDERWATER) ) )
	{
		//R_RenderDynamicLightmaps (s);
		if (gl_mtexable) {
			p = s->polys;

			t = dr_R_TextureAnimation (s->texinfo->texture);
			// Binds world to texture env 0
			GL_SelectTexture(TEXTURE0_SGIS, gl_mtexable, qglSelectTextureSGIS,
						currenttexture, cnttextures );
			dr_GL_Bind (t->gl_texturenum);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
			// Binds lightmap to texenv 1
			dr_GL_DisableMultitexture(); // Same as SelectTexture (TEXTURE1)
			dr_GL_Bind (lightmap_textures + s->lightmaptexturenum);
			i = s->lightmaptexturenum;
			if (lightmap_modified[i])
			{
				lightmap_modified[i] = false;
				theRect = &lightmap_rectchange[i];
				glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t,
					BLOCK_WIDTH, theRect->h, gl_lightmap_format, GL_UNSIGNED_BYTE,
					lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*lightmap_bytes);
				theRect->l = BLOCK_WIDTH;
				theRect->t = BLOCK_HEIGHT;
				theRect->h = 0;
				theRect->w = 0;
			}
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_BLEND);
			glBegin(GL_POLYGON);
			v = p->verts[0];
			for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
			{
				qglMTexCoord2fSGIS (TEXTURE0_SGIS, v[3], v[4]);
				qglMTexCoord2fSGIS (TEXTURE1_SGIS, v[5], v[6]);
				glVertex3fv (v);
			}
			glEnd ();
			return;
		} else {
			p = s->polys;

			t = dr_R_TextureAnimation (s->texinfo->texture);
			dr_GL_Bind (t->gl_texturenum);
			glBegin (GL_POLYGON);
			v = p->verts[0];
			for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
			{
				glTexCoord2f (v[3], v[4]);
				glVertex3fv (v);
			}
			glEnd ();
/*
			dr_GL_Bind (lightmap_textures + s->lightmaptexturenum);
			glEnable (GL_BLEND);
			glBegin (GL_POLYGON);
			v = p->verts[0];
			for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
			{
				glTexCoord2f (v[5], v[6]);
				glVertex3fv (v);
			}
			glEnd ();
*/
			glDisable (GL_BLEND);
		}

		return;
	}

	//
	// subdivided water surface warp
	//

	if (s->flags & SURF_DRAWTURB)
	{
		dr_GL_DisableMultitexture();
		dr_GL_Bind (s->texinfo->texture->gl_texturenum);
		EmitWaterPolys( s, realtime );
		return;
	}

	//
	// subdivided sky warp
	//
	if (s->flags & SURF_DRAWSKY)
	{
		dr_GL_DisableMultitexture();
		dr_GL_Bind (solidskytexture);
		*speedscale = realtime*8;
		*speedscale -= (int)(*speedscale) & ~127;

		EmitSkyPolys (s, *speedscale, r_origin);

		glEnable (GL_BLEND);
		dr_GL_Bind (alphaskytexture);
		*speedscale = realtime*16;
		*speedscale -= (int)(*speedscale) & ~127;
		EmitSkyPolys (s, *speedscale, r_origin);

		glDisable (GL_BLEND);
		return;
	}

	//
	// underwater warped with lightmap
	//
	//R_RenderDynamicLightmaps (s);
	if (gl_mtexable) {
		p = s->polys;

		t = dr_R_TextureAnimation (s->texinfo->texture);
			GL_SelectTexture(TEXTURE0_SGIS, gl_mtexable, qglSelectTextureSGIS,
				currenttexture, cnttextures );
		dr_GL_Bind (t->gl_texturenum);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
		dr_GL_DisableMultitexture();
		dr_GL_Bind (lightmap_textures + s->lightmaptexturenum);
		i = s->lightmaptexturenum;
		if (lightmap_modified[i])
		{
			lightmap_modified[i] = false;
			theRect = &lightmap_rectchange[i];
//			glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t,
//				BLOCK_WIDTH, theRect->h, gl_lightmap_format, GL_UNSIGNED_BYTE,
//				lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*lightmap_bytes);
			theRect->l = BLOCK_WIDTH;
			theRect->t = BLOCK_HEIGHT;
			theRect->h = 0;
			theRect->w = 0;
		}
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_BLEND);
		glBegin (GL_TRIANGLE_FAN);
		v = p->verts[0];
		for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
		{
			qglMTexCoord2fSGIS (TEXTURE0_SGIS, v[3], v[4]);
			qglMTexCoord2fSGIS (TEXTURE1_SGIS, v[5], v[6]);

			nv[0] = v[0] + 8*sin(v[1]*0.05+realtime)*sin(v[2]*0.05+realtime);
			nv[1] = v[1] + 8*sin(v[0]*0.05+realtime)*sin(v[2]*0.05+realtime);
			nv[2] = v[2];

			glVertex3fv (nv);
		}
		glEnd ();

	} else {
		p = s->polys;

		t = dr_R_TextureAnimation (s->texinfo->texture);
		dr_GL_Bind (t->gl_texturenum);
		DrawGLWaterPoly( p, realtime );

		dr_GL_Bind (lightmap_textures + s->lightmaptexturenum);
		glEnable (GL_BLEND);
		DrawGLWaterPolyLightmap( p, realtime );
		glDisable (GL_BLEND);
	}
}


/*
================
DrawGLWaterPoly

Warp the vertex coordinates
================
*/
EXPORT void DrawGLWaterPoly (glpoly_t *p, double realtime)
{
	int		i;
	float	*v;
	vec3_t	nv;

	glPushAttrib(GL_ALL_ATTRIB_BITS);

	//dr_GL_DisableMultitexture();
	glEnable(GL_TEXTURE_2D);

	glEnable( GL_POLYGON_OFFSET_FILL );
	glPolygonOffset( 1.0, 1.0 );

		glPolygonMode(GL_BACK, GL_FILL);
	setColor(tech_water_color.value);
	glBegin (GL_TRIANGLE_FAN);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		nv[0] = v[0] + 8*sin(v[1]*0.05+realtime)*sin(v[2]*0.05+realtime);
		nv[1] = v[1] + 8*sin(v[0]*0.05+realtime)*sin(v[2]*0.05+realtime);
		nv[2] = v[2];

		glVertex3fv (nv);
	}
	glEnd ();

	glPolygonMode(GL_BACK, GL_LINE);
	setColor(tech_wall_line_color.value);
	glLineWidth(tech_line_width.value);
	glBegin (GL_TRIANGLE_FAN);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		nv[0] = v[0] + 8*sin(v[1]*0.05+realtime)*sin(v[2]*0.05+realtime);
		nv[1] = v[1] + 8*sin(v[0]*0.05+realtime)*sin(v[2]*0.05+realtime);
		nv[2] = v[2];

		glVertex3fv (nv);
	}
	glEnd ();

	glPopAttrib();

}


//adapted from sketch NPR Quake (took out randomness)
EXPORT void DrawGLWaterPolyLightmap (glpoly_t *p, double realtime)
{
	int		i;
	float	*v;
	vec3_t	nv;
	int j,k;

	dr_GL_DisableMultitexture();

	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);
	glBlendFunc (GL_ZERO,GL_ONE_MINUS_SRC_COLOR);

	glBegin (GL_TRIANGLE_FAN);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		glTexCoord2f (v[5], v[6]);

		nv[0] = v[0] + 8*sin(v[1]*0.05+realtime)*sin(v[2]*0.05+realtime);
		nv[1] = v[1] + 8*sin(v[0]*0.05+realtime)*sin(v[2]*0.05+realtime);
		nv[2] = v[2];

		glVertex3fv (nv);
	}
	glEnd ();

	CurRand = 0;

	glDisable(GL_TEXTURE_2D);
	glEnable(GL_BLEND) ;
	glBlendFunc (GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);

	glLineWidth(tech_line_width.value) ;
	glColor4f(0.0f, 0.0f, 0.0f, 0.25);

	for (j=0; j < 6; j++)
	{
		glBegin (GL_LINE_LOOP) ;
		v = p->verts[0];
		for (k=0 ; k<p->numverts ; k++, v+= VERTEXSIZE)
		{
			glVertex3f (v[0] + randFloat (-1.0f, 1.0f), v[1] + randFloat (-1.0f, 1.0f), v[2] + randFloat (-1.0f, 1.0f)) ;
		}
		glEnd ();
	}
	glLineWidth(1.0) ;
}

/*
================
DrawGLPoly
================
*/
//adapted from sketch NPR Quake (took out randomness)
EXPORT void DrawGLPoly (glpoly_t *p)
{
	int		i;
	float	*v;
	float vert[4] = { 0.0f,0.0f,0.8f,0.8f};
	glPushAttrib(GL_ALL_ATTRIB_BITS);

	glEnable( GL_POLYGON_OFFSET_FILL );
	glPolygonOffset( 1.0, 1.0 );

	glDisable(GL_TEXTURE_2D);
	glDisable(GL_TEXTURE_1D);
	//dr_GL_Bind(texNum[(int)ainpr_tex_no.value]);

	glLightfv(GL_LIGHT0,GL_DIFFUSE,(GLfloat *)&vert);
	glLightfv(GL_LIGHT0,GL_SPECULAR,(GLfloat *)&vert);
	glLightfv(GL_LIGHT0,GL_AMBIENT,(GLfloat *)&vert);
	glLightfv(GL_LIGHT0,GL_POSITION,(GLfloat *)&vert);
	setColor(tech_wall_color.value);
	glBegin (GL_POLYGON);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		//glTexCoord2f (v[3], v[4]);
		//glTexCoord1f (v[3] * v[4]);

		glVertex3fv (v);
	}
	glEnd ();

	glLineWidth(tech_line_width.value);
	setColor(tech_wall_line_color.value);
	glBegin(GL_LINE_LOOP);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		glTexCoord2f (v[3], v[4]);
		glVertex3fv (v);
	}
	glEnd ();

	glPopAttrib();

}

/*
================
R_BlendLightmaps
================
*/
//adapted from sketch NPR Quake (took out randomness)
EXPORT void R_BlendLightmaps ( glpoly_t ** lightmap_polys, int lightmap_textures,
					   qboolean * lightmap_modified, glRect_t * lightmap_rectchange,
					   byte * lightmaps, int lightmap_bytes, int gl_lightmap_format,
					   cvar_t * r_lightmap, cvar_t * gl_texsort, cvar_t * r_fullbright,
					   double realtime)
{
	int			i, j, k;
	glpoly_t	*p;
	float		*v;

	if (r_fullbright->value)
		return;
	if (!gl_texsort->value)
		return;

	glDepthMask (0);		// don't bother writing Z

	if (gl_lightmap_format == GL_LUMINANCE)
		glBlendFunc (GL_ZERO, GL_ONE_MINUS_SRC_COLOR);
	else if (gl_lightmap_format == GL_INTENSITY)
	{
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		glColor4f (0,0,0,1);
		glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}

	if (!r_lightmap->value)
	{
		glEnable (GL_BLEND);
	}

	for (i=0 ; i<MAX_LIGHTMAPS ; i++)
	{
		p = lightmap_polys[i];
		if (!p)
			continue;
		dr_GL_Bind(lightmap_textures+i);

		for ( ; p ; p=p->chain)
		{
			if (p->flags & SURF_UNDERWATER)
				DrawGLWaterPolyLightmap( p, realtime );
			else
			{
				glBlendFunc (GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
				glEnable(GL_TEXTURE_2D);
				glBegin (GL_POLYGON);
				v = p->verts[0];
				for (j=0 ; j<p->numverts ; j++, v+= VERTEXSIZE)
				{
					glTexCoord2f (v[5], v[6]);
					glVertex3fv (v);
				}
				glEnd ();

				CurRand = 0;

				glColor4f(0.0f, 0.0f, 0.0f, 0.25);

				glBlendFunc (GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
				glDisable(GL_TEXTURE_2D);
				for (j=0; j < 6; j++)
				{
					glBegin (GL_LINE_LOOP) ;

					v = p->verts[0];
					for (k=0 ; k<p->numverts ; k++, v+= VERTEXSIZE)
					{
						glVertex3f (v[0] + randFloat (-1.0f, 1.0f), v[1] + randFloat (-1.0f, 1.0f), v[2] + randFloat (-1.0f, 1.0f)) ;
					}
					glEnd ();
				}
			}
		}
	}

	glDisable (GL_BLEND);
	if (gl_lightmap_format == GL_LUMINANCE)
		glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	else if (gl_lightmap_format == GL_INTENSITY)
	{
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
		glColor4f (1,1,1,1);
	}

	glDepthMask (1);		// back to normal Z buffering
}

/*
===============
R_DrawParticles
===============
*/
//extern	cvar_t	sv_gravity;

EXPORT void R_DrawParticles (particle_t ** active_particles, particle_t ** free_particles,
					  int * ramp1, int * ramp2, int * ramp3, cvar_t * sv_gravity,
					  double cloldtime, double cltime, int particletexture,
					  vec3_t vright, vec3_t vup, unsigned int * d_8to24table,
					  vec3_t vpn, vec3_t r_origin)
{
	particle_t		*p, *kill;
	float			grav;
	int				i;
	float			time2, time3;
	float			time1;
	float			dvel;
	float			frametime;

	vec3_t			up, right;
	float			scale;

	dr_GL_Bind(particletexture);
	glEnable (GL_BLEND);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glBegin (GL_TRIANGLES);

	dr_VectorScale (vup, 1.5, up);
	dr_VectorScale (vright, 1.5, right);

	frametime = cltime - cloldtime;
	time3 = frametime * 15;
	time2 = frametime * 10; // 15;
	time1 = frametime * 5;
	grav = frametime * sv_gravity->value * 0.05;
	dvel = 4*frametime;

	for ( ;; )
	{
		kill = (*active_particles);
		if (kill && kill->die < cltime)
		{
			(*active_particles) = kill->next;
			kill->next = (*free_particles);
			(*free_particles) = kill;
			continue;
		}
		break;
	}

	for (p=(*active_particles) ; p ; p=p->next)
	{
		for ( ;; )
		{
			kill = p->next;
			if (kill && kill->die < cltime)
			{
				p->next = kill->next;
				kill->next = (*free_particles);
				(*free_particles) = kill;
				continue;
			}
			break;
		}

		// hack a scale up to keep particles from disapearing
		scale = (p->org[0] - r_origin[0])*vpn[0] + (p->org[1] - r_origin[1])*vpn[1]
			+ (p->org[2] - r_origin[2])*vpn[2];
		if (scale < 20)
			scale = 1;
		else
			scale = 1 + scale * 0.004;



		glColor3ubv ((byte *)&(d_8to24table[(int)p->color]));


		glTexCoord2f (0,0);
		glVertex3fv (p->org);
		glTexCoord2f (1,0);
		glVertex3f (p->org[0] + up[0]*scale, p->org[1] + up[1]*scale, p->org[2] + up[2]*scale);
		glTexCoord2f (0,1);
		glVertex3f (p->org[0] + right[0]*scale, p->org[1] + right[1]*scale, p->org[2] + right[2]*scale);

		p->org[0] += p->vel[0]*frametime;
		p->org[1] += p->vel[1]*frametime;
		p->org[2] += p->vel[2]*frametime;



		switch (p->type)
		{
		case pt_static:
			break;
		case pt_fire:
			p->ramp += time1;
			if (p->ramp >= 6)
				p->die = -1;
			else
				p->color = ramp3[(int)p->ramp];
			p->vel[2] += grav;
			break;

		case pt_explode:
			p->ramp += time2;
			if (p->ramp >=8)
				p->die = -1;
			else
				p->color = ramp1[(int)p->ramp];
			for (i=0 ; i<3 ; i++)
				p->vel[i] += p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_explode2:
			p->ramp += time3;
			if (p->ramp >=8)
				p->die = -1;
			else
				p->color = ramp2[(int)p->ramp];
			for (i=0 ; i<3 ; i++)
				p->vel[i] -= p->vel[i]*frametime;
			p->vel[2] -= grav;
			break;

		case pt_blob:
			for (i=0 ; i<3 ; i++)
				p->vel[i] += p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_blob2:
			for (i=0 ; i<2 ; i++)
				p->vel[i] -= p->vel[i]*dvel;
			p->vel[2] -= grav;
			break;

		case pt_grav:

		case pt_slowgrav:
			p->vel[2] -= grav;
			break;
		}

	}

	glEnd ();
	glDisable (GL_BLEND);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

}

/*
=============
EmitWaterPolys

Does a water warp on the pre-fragmented glpoly_t chain
=============
*/
//adapted from sketch NPR Quake (took out randomness)
EXPORT void EmitWaterPolys (msurface_t *fa, double realtime)
{
	glpoly_t	*p;
	float		*v;
	int			i;
	int			j;
	float		s, t, os, ot;

	glDisable(GL_TEXTURE_2D) ;
	glEnable(GL_BLEND) ;
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA) ;
	glDepthMask( 1 );

	glColor4f (0.34f,0.52f,0.75f, 1.0f) ;

	for (p=fa->polys ; p ; p=p->next)
	{
		glBegin (GL_POLYGON);
		for (i=0,v=p->verts[0] ; i<p->numverts ; i++, v+=VERTEXSIZE)
		{
			glVertex3fv (v);
		}
		glEnd ();
	}

	glDepthMask( 0 );

	for (p=fa->polys ; p ; p=p->next)
	{
		CurRand = 0;

		//glColor4f (0.64f,0.82f,0.95f,ainpr_alpha.value);
		glColor4f (0.0f,0.0f,0.0f,tech_shadow_alpha.value);

		glLineWidth(tech_line_width.value) ;
		glBegin (GL_LINE_LOOP);
		for (j=0; j < 6; j++)
		{
			for (i=0,v=p->verts[0] ; i<p->numverts ; i++, v+=VERTEXSIZE)
			{
				os = v[0];
				ot = v[1];

				s = 15 * turbsin[(int)((ot*0.125+realtime) * 300) & 255];
				s *= (1.0/64);

				t = 15 * turbsin[(int)((os*0.125+realtime) * 300) & 255];
				t *= (1.0/64);

				glVertex3f (v[0]+s+randFloat(-1.0,1.0), v[1]+t+randFloat(-1.0,1.0), v[2]+randFloat(-1.0,1.0)) ;
			}
			glLineWidth(1) ;
		}
		glEnd ();
	}

	glLineWidth(1.0) ;
	glDepthMask (1);
	glDisable(GL_BLEND) ;
	glEnable(GL_TEXTURE_2D) ;
}

/*
=============
EmitSkyPolys
=============
*/
EXPORT void EmitSkyPolys (msurface_t *fa, float speedscale, vec3_t r_origin)
{
	glpoly_t	*p;
	float		*v;
	int			i;
	float	s, t;
	vec3_t	dir;
	float	length;

	for (p=fa->polys ; p ; p=p->next)
	{
		glBegin (GL_POLYGON);
		for (i=0,v=p->verts[0] ; i<p->numverts ; i++, v+=VERTEXSIZE)
		{
			VectorSubtract (v, r_origin, dir);
			dir[2] *= 3;	// flatten the sphere

			length = dir[0]*dir[0] + dir[1]*dir[1] + dir[2]*dir[2];
			length = sqrt (length);
			length = 6*63/length;

			dir[0] *= length;
			dir[1] *= length;

			s = (speedscale + dir[0]) * (1.0/128);
			t = (speedscale + dir[1]) * (1.0/128);

			glTexCoord2f (s, t);
			glVertex3fv (v);
		}
		glEnd ();
	}
}
